import json
import math
import multiprocessing
import time

from wikibaseintegrator import WikibaseIntegrator
from wikibaseintegrator.wbi_config import config
from wikibaseintegrator.wbi_helpers import (
    execute_sparql_query,
    mediawiki_api_call_helper,
)
from wikibaseintegrator.wbi_login import Login

from config.secrets import password, user
from utils.csv_logger import Logger
from utils.utils import statement_redirect_query_result_to_edit_list


# Het aantal processen dat je wilt gebruiken (voor snellere werking)
N_PROCESSES = 2

# De nodige configuratie voor onze wikibase instantie
config['DEFAULT_LANGUAGE'] = 'nl'
config['WIKIBASE_URL'] = 'https://kg.kunsten.be'
config['MEDIAWIKI_API_URL'] = 'https://kg.kunsten.be/w/api.php'
config['MEDIAWIKI_INDEX_URL'] = 'https://kg.kunsten.be/w/index.php'
config['MEDIAWIKI_REST_URL'] = 'https://kg.kunsten.be/w/rest.php'
config['SPARQL_ENDPOINT_URL'] = (
    'https://kg.kunsten.be/query/proxy/wdqs/bigdata/namespace/wdq/sparql'
)
# De login van de bot account (niet zomaar online zetten / publiek delen!)
login = Login(user=user, password=password)
wbi = WikibaseIntegrator(login=login)

query = """
SELECT DISTINCT ?subject ?old ?new
WHERE {
  ?old owl:sameAs ?new .
  ?subject ?predicate ?old .
  ?subject wikibase:rank [] .
}
"""

query_result = execute_sparql_query(query)
edit_list = statement_redirect_query_result_to_edit_list(query_result)

def task(edit_list, process_index):
    # De login van de bot account (niet zomaar online zetten / publiek delen!)
    login = Login(user=user, password=password)
    wbi = WikibaseIntegrator(login=login, is_bot=True)

    logger = Logger(f'fix_statement_redirects_errors_process_{process_index}.csv')
    logger.write_row(['statement_id', 'old_id', 'new_id', 'error_type', 'error_message'])

    total = len(edit_list)

    print(f'Starting process {process_index} with edit list of length {total}')

    for index, [statement_id, old_id, new_id] in enumerate(edit_list):
        print(
            f'Process {process_index}: Changing statement {statement_id} to accomodate redirect: {old_id} -> {new_id} ({index + 1}/{total})'
        )

        params = {
            'action': 'wbgetclaims',
            'claim': statement_id,
        }
        result = mediawiki_api_call_helper(data=params, login=login, is_bot=True)
        result_json = json.dumps(list(result['claims'].values())[0][0])
        # Het lijkt alsof enkel de numeric-id aanpassen ook voldoende is,
        # maar voor het zeker toch beide nog doen
        updated_json = result_json.replace(f'"{old_id}"', f'"{new_id}"')
        updated_json = updated_json.replace(f'"numeric-id": {old_id[1:]}', f'"numeric-id": {new_id[1:]}')

        params = {
            'action': 'wbsetclaim',
            'claim': updated_json,
            'summary': 'update value to accomodate redirect',
        }
        try:
            mediawiki_api_call_helper(data=params, login=login, is_bot=True)
        except Exception as err:
            logger.write_row([statement_id, old_id, new_id, type(err), err])
            print(err)

window_size = math.ceil(len(edit_list) / N_PROCESSES)
processes = []
start = time.time()
for i in range(N_PROCESSES):
    p = multiprocessing.Process(target=task, args=(edit_list[(i * window_size):(i * window_size + window_size)],i))
    p.start()
    processes.append(p)
    # Nodig zodat wikibase niet klaagt dat we te snel meerdere keren aan het inloggen zijn :)
    time.sleep(0.2)
    
for p in processes:
    p.join()

print(f'Processing of {len(edit_list)} statements done in {time.time() - start} seconds over {N_PROCESSES} processes.')